<?php
/* =========================
   VIP-PRO Movie Report (functions.php)
   - Không phải plugin, dán vào functions.php
   - Lưu postmeta key: "video_report"
   - AJAX modal, nonce, honeypot, spam-limit (transient), admin page, webhook
   - Admin menu badge = số report pending
   ========================= */

if (! defined('ABSPATH')) exit;

/* --- Constants & Defaults --- */
if (! defined('VPMR_META_KEY')) define('VPMR_META_KEY', 'video_report');
if (! defined('VPMR_OPTION_KEY')) define('VPMR_OPTION_KEY', 'vpmr_options');

function vpmr_get_defaults()
{
  return array(
    'spam_limit_count' => 1,
    'spam_limit_window' => 86400, // seconds
    'webhook_url' => '',
    'enable_webhook' => 0,
  );
}

/* Ensure defaults exist */
add_action('after_setup_theme', function () {
  if (false === get_option(VPMR_OPTION_KEY)) {
    update_option(VPMR_OPTION_KEY, vpmr_get_defaults());
  }
});

/* --- Enqueue Frontend (only on single posts) --- */
add_action('wp_enqueue_scripts', function () {
  if (! is_singular()) return;
  // register empty handles to attach inline code
  wp_register_style('vpmr-inline', false);
  wp_enqueue_style('vpmr-inline');
  wp_add_inline_style('vpmr-inline', "
/* vpmr minimal modal */
#vpmr-modal{position:fixed;inset:0;display:none;align-items:center;justify-content:center;z-index:99999}
.vpmr-backdrop{position:absolute;inset:0;background:rgba(0,0,0,.5)}
.vpmr-panel{border:solid 1.2px #161616;position:relative;z-index:2;background:#000;padding:18px;max-width:520px;width:94%;box-shadow:0 10px 30px rgba(0,0,0,.15)}
.vpmr-close{position:absolute;right:10px;top:6px;border:0;background:transparent;font-size:22px;cursor:pointer}
.vpmr-panel select,.vpmr-panel textarea{width:100%;max-width:100%;padding:8px;background:#111;border: 1.2px solid #2a2a2a;color:#eee;margin-top:8px}
.vpmr-submit{display:inline-block;padding:8px 14px;border:0;background:var(--color-primary);color:#fff;cursor:pointer;margin-top:10px}
.vpmr-feedback{margin-top:8px;padding:8px;background:#161616;display:none}
");

  wp_register_script('vpmr-frontend', false, array(), false, true);
  wp_enqueue_script('vpmr-frontend');

  $data = array(
    'ajax_url' => admin_url('admin-ajax.php'),
    'nonce'    => wp_create_nonce('vpmr_report_nonce'),
    'post_id'  => get_the_ID(),
    'strings'  => array(
      'thanks' => 'Thanks for reporting the error, we will check it.',
      'already' => 'You have reported this video for a short time. Please try again later.',
      'err' => 'An error occurred. Please try again.'
    )
  );
  wp_localize_script('vpmr-frontend', 'vpmr_data', $data);

  // Inline JS (minimal, no external file)
  wp_add_inline_script('vpmr-frontend', "
(function(){
  document.addEventListener('click', function(e){
    var btn = e.target.closest && e.target.closest('.action-btn.report, .vpmr-open-report');
    if (btn){
      e.preventDefault();
      var modal = document.getElementById('vpmr-modal');
      if (modal) modal.style.display = 'flex';
      var fb = modal && modal.querySelector('.vpmr-feedback');
      if (fb) fb.style.display = 'none';
    }
    if (e.target.closest && e.target.closest('.vpmr-close')) {
      var modal = document.getElementById('vpmr-modal');
      if (modal) modal.style.display = 'none';
    }
    if (e.target.closest && e.target.closest('.vpmr-backdrop')) {
      var modal = document.getElementById('vpmr-modal');
      if (modal) modal.style.display = 'none';
    }
  }, false);

  document.addEventListener('submit', function(e){
    if (e.target && e.target.id === 'vpmr-form') {
      e.preventDefault();
      var form = e.target;
      var btn = form.querySelector('.vpmr-submit');
      if (btn) btn.disabled = true;
      var data = new FormData(form);
      data.append('action','vpmr_submit_report');
      data.append('nonce', vpmr_data.nonce);

      fetch(vpmr_data.ajax_url, {
        method:'POST',
        credentials:'same-origin',
        body: data
      }).then(function(resp){
        return resp.json().then(function(json){ return { status: resp.status, json: json }; });
      }).then(function(r){
        var fb = form.querySelector('.vpmr-feedback');
        if (r.status >=200 && r.status < 300 && r.json.success) {
          if (fb) { fb.style.display='block'; fb.innerText = vpmr_data.strings.thanks; }
          form.reset();
          setTimeout(function(){ var modal=document.getElementById('vpmr-modal'); if(modal) modal.style.display='none'; }, 1100);
          // optional: update UI count badge on frontend if exists
          var badge = document.querySelector('.vpmr-admin-badge-count');
          if (badge) {
            var n = parseInt(badge.innerText||'0',10); badge.innerText = n+1;
            badge.style.display='inline-block';
          }
        } else {
          var msg = (r.json && r.json.data && r.json.data.message) ? r.json.data.message : 'error';
          if (msg === 'limit') {
            if (fb) fb.style.display='block', fb.innerText = vpmr_data.strings.already;
          } else {
            if (fb) fb.style.display='block', fb.innerText = vpmr_data.strings.err;
          }
        }
      }).catch(function(){
        var fb = form.querySelector('.vpmr-feedback');
        if (fb) fb.style.display='block', fb.innerText = 'Network error';
      }).finally(function(){ if (btn) btn.disabled = false; });
    }
  }, false);
})();
");
});

/* --- Modal HTML (print in footer on single posts) --- */
add_action('wp_footer', function () {
  if (! is_singular()) return;
?>
  <div id="vpmr-modal" style="display:none;align-items:center;justify-content:center;">
    <div class="vpmr-backdrop"></div>
    <div class="vpmr-panel" role="dialog" aria-modal="true">
      <button class="vpmr-close" aria-label="Close">&times;</button>
      <h3 style="margin:0 0 8px 0">Video Error Report</h3>
      <form id="vpmr-form">
        <?php wp_nonce_field('vpmr_report_nonce', 'vpmr_nonce_field', false); ?>
        <input type="text" name="vpmr_hp" value="" style="position:absolute;left:-9999px;top:auto;opacity:0" autocomplete="off">
        <input type="hidden" name="post_id" value="<?php echo esc_attr(get_the_ID()); ?>">
        <label>Reason
          <select name="reason" required>
            <option value="">Select Reason</option>
            <option value="video_error">Video Error</option>
            <option value="audio_error">Audio Error</option>
            <option value="subtitle_wrong">Subtitle Error</option>
            <option value="not_playing">Video Not Playing</option>
            <option value="other">Other</option>
          </select>
        </label>
        <label>Description (Optional)
          <textarea name="message" rows="4" placeholder="For example: server 2 die,..."></textarea>
        </label>
        <button class="vpmr-submit" type="submit">Submit</button>
        <div class="vpmr-feedback"></div>
      </form>
    </div>
  </div>
<?php
}, 20);

/* --- Helper: Get client IP (best-effort) --- */
function vpmr_get_client_ip()
{
  if (! empty($_SERVER['HTTP_CF_CONNECTING_IP'])) return sanitize_text_field(wp_unslash($_SERVER['HTTP_CF_CONNECTING_IP']));
  if (! empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
    $list = explode(',', wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR']));
    return sanitize_text_field(trim($list[0]));
  }
  return isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '0.0.0.0';
}

/* --- AJAX handler: submit report --- */
add_action('wp_ajax_vpmr_submit_report', 'vpmr_submit_report');
add_action('wp_ajax_nopriv_vpmr_submit_report', 'vpmr_submit_report');

function vpmr_submit_report()
{
  // nonce
  if (empty($_POST['nonce']) || ! wp_verify_nonce(sanitize_key($_POST['nonce']), 'vpmr_report_nonce')) {
    wp_send_json_error(array('message' => 'invalid_nonce'), 400);
  }

  // honeypot
  if (! empty($_POST['vpmr_hp'])) {
    wp_send_json_error(array('message' => 'spam'), 400);
  }

  $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
  if ($post_id <= 0) wp_send_json_error(array('message' => 'invalid_post'), 400);

  $reason = isset($_POST['reason']) ? sanitize_text_field(wp_unslash($_POST['reason'])) : '';
  $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';

  $opts = get_option(VPMR_OPTION_KEY, vpmr_get_defaults());
  $limit_count = max(1, intval($opts['spam_limit_count']));
  $limit_window = max(60, intval($opts['spam_limit_window']));

  $ip = vpmr_get_client_ip();
  $transient_key = 'vpmr_' . md5($ip . '::' . $post_id);

  $count = intval(get_transient($transient_key));
  if ($count >= $limit_count) {
    wp_send_json_error(array('message' => 'limit'), 429);
  }

  $report = array(
    'time'    => current_time('mysql'),
    'ip'      => $ip,
    'reason'  => $reason,
    'message' => $message,
    'user_id' => get_current_user_id(),
    'status'  => 'pending',
    'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : ''
  );

  // store as multiple postmeta entries (one report per add_post_meta)
  $added = add_post_meta($post_id, VPMR_META_KEY, $report);

  if ($added) {
    set_transient($transient_key, $count + 1, $limit_window);

    // webhook (non-blocking best-effort)
    if (! empty($opts['enable_webhook']) && ! empty($opts['webhook_url'])) {
      $body = array(
        'action' => 'vpmr_new_report',
        'post_id' => $post_id,
        'post_title' => get_the_title($post_id),
        'report' => $report,
      );
      // try wp_remote_post with tiny timeout, non-blocking if possible
      wp_remote_post(esc_url_raw($opts['webhook_url']), array(
        'timeout' => 2,
        'blocking' => false,
        'body' => wp_json_encode($body),
        'headers' => array('Content-Type' => 'application/json')
      ));
    }

    wp_send_json_success(array('message' => 'ok'), 200);
  } else {
    wp_send_json_error(array('message' => 'db_error'), 500);
  }
}

/* --- Admin: menu + badge (pending count) --- */
add_action('admin_menu', 'vpmr_admin_menu');
function vpmr_admin_menu()
{
  $count = vpmr_get_pending_count();
  $title = 'Movie Reports' . ($count > 0 ? " <span class='vpmr-badge' style='color:#fff;background:#d9534f;border-radius:10px;padding:2px 6px;margin-left:8px;font-size:12px;vertical-align:middle'>{$count}</span>" : '');
  add_menu_page('Movie Reports', $title, 'manage_options', 'vpmr_reports', 'vpmr_admin_page', 'dashicons-warning', 58);
  add_submenu_page('vpmr_reports', 'Settings', 'Settings', 'manage_options', 'vpmr_reports_settings', 'vpmr_settings_page');

  // add small inline style for badge in admin menu area (safe)
  add_action('admin_head', function () {
    echo '<style>.toplevel_page_vpmr_reports .vpmr-badge{display:inline-block}</style>';
  });
}

/* Count pending reports (scan postmeta for key and check status) */
function vpmr_get_pending_count()
{
  global $wpdb;
  $meta_table = $wpdb->postmeta;
  $like = '%\"status\";s:%\"pending\"%'; // not used directly; we'll do simple LIKE for "pending"
  $sql = $wpdb->prepare("
        SELECT COUNT(*) FROM {$meta_table} pm
        WHERE pm.meta_key = %s AND pm.meta_value LIKE %s
    ", VPMR_META_KEY, '%' . $wpdb->esc_like('\"status\";s:') . '%' . $wpdb->esc_like('pending') . '%');
  $count = intval($wpdb->get_var($sql));
  return $count;
}

/* --- Admin: Reports Page --- */
function vpmr_admin_page()
{
  if (! current_user_can('manage_options')) wp_die('Permission denied');
  global $wpdb;
  $meta_table = $wpdb->postmeta;

  $paged = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
  $per_page = 25;
  $offset = ($paged - 1) * $per_page;

  $status = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';

  $where_status_sql = '';
  if (in_array($status, array('pending', 'resolved'), true)) {
    $where_status_sql = $wpdb->prepare(" AND pm.meta_value LIKE %s ", '%' . $wpdb->esc_like('"status";s:') . '%' . $wpdb->esc_like($status) . '%');
  }

  $total = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$meta_table} pm
        WHERE pm.meta_key = %s {$where_status_sql}
    ", VPMR_META_KEY));

  $rows = $wpdb->get_results($wpdb->prepare("
        SELECT pm.meta_id, pm.post_id, pm.meta_value
        FROM {$meta_table} pm
        WHERE pm.meta_key = %s {$where_status_sql}
        ORDER BY pm.meta_id DESC
        LIMIT %d OFFSET %d
    ", VPMR_META_KEY, $per_page, $offset), ARRAY_A);

?>
  <div class="wrap">
    <h1>Báo Lỗi Videos</h1>
    <p>
      <a href="<?php echo esc_url(admin_url('admin.php?page=vpmr_reports_settings')); ?>" class="button">Settings</a>
    </p>

    <form method="get" style="margin-bottom:1em;">
      <input type="hidden" name="page" value="vpmr_reports">
      <select name="status">
        <option value="">All statuses</option>
        <option value="pending" <?php selected($status, 'pending'); ?>>Pending</option>
        <option value="resolved" <?php selected($status, 'resolved'); ?>>Resolved</option>
      </select>
      <button class="button">Filter</button>
    </form>

    <table class="widefat striped">
      <thead>
        <tr>
          <th>ID</th>
          <th>Movie</th>
          <th>Reason</th>
          <th>Message</th>
          <th>Time</th>
          <th>IP / User</th>
          <th>Status</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php if (empty($rows)) : ?>
          <tr>
            <td colspan="8">No reports found.</td>
          </tr>
          <?php else: foreach ($rows as $row):
            $meta_id = intval($row['meta_id']);
            $post_id = intval($row['post_id']);
            $data = maybe_unserialize($row['meta_value']);
            if (! is_array($data)) continue;
          ?>
            <tr>
              <td><?php echo esc_html($meta_id); ?></td>
              <td><a href="<?php echo esc_url(get_edit_post_link($post_id)); ?>" target="_blank"><?php echo esc_html(get_the_title($post_id)); ?></a></td>
              <td><?php echo esc_html($data['reason'] ?? ''); ?></td>
              <td style="max-width:300px;word-break:break-word;"><?php echo esc_html($data['message'] ?? ''); ?></td>
              <td><?php echo esc_html($data['time'] ?? ''); ?></td>
              <td><?php echo esc_html(substr($data['ip'] ?? '', 0, 30)); ?> / <?php echo intval($data['user_id'] ?? 0); ?></td>
              <td><?php echo esc_html($data['status'] ?? 'pending'); ?></td>
              <td>
                <?php if (($data['status'] ?? 'pending') !== 'resolved'): ?>
                  <a class="button" href="<?php echo esc_url(wp_nonce_url(admin_url('admin-post.php?action=vpmr_resolve&meta_id=' . $meta_id), 'vpmr_resolve_' . $meta_id)); ?>">Mark Resolved</a>
                <?php endif; ?>
                <a class="button" style="background:#a00;border-color:#900;color:#fff" href="<?php echo esc_url(wp_nonce_url(admin_url('admin-post.php?action=vpmr_delete&meta_id=' . $meta_id), 'vpmr_delete_' . $meta_id)); ?>">Delete</a>
              </td>
            </tr>
        <?php endforeach;
        endif; ?>
      </tbody>
    </table>

    <?php
    $base = add_query_arg('page', 'vpmr_reports', admin_url('admin.php'));
    $total_pages = ceil($total / $per_page);
    if ($total_pages > 1):
      echo '<p class="tablenav"><span class="pagination-links">';
      for ($i = 1; $i <= $total_pages; $i++) {
        $url = add_query_arg(array('page' => 'vpmr_reports', 'paged' => $i, 'status' => $status), admin_url('admin.php'));
        echo '<a class="' . ($i === $paged ? 'page-numbers current' : 'page-numbers') . '" href="' . esc_url($url) . '">' . $i . '</a> ';
      }
      echo '</span></p>';
    endif;
    ?>
  </div>
<?php
}

/* --- Admin actions: resolve / delete via admin-post --- */
add_action('admin_post_vpmr_resolve', function () {
  if (! current_user_can('manage_options')) wp_die('permission');
  $meta_id = isset($_GET['meta_id']) ? intval($_GET['meta_id']) : 0;
  if (! wp_verify_nonce($_GET['_wpnonce'] ?? '', 'vpmr_resolve_' . $meta_id)) wp_die('nonce');
  global $wpdb;
  $meta = $wpdb->get_row($wpdb->prepare("SELECT meta_value FROM {$wpdb->postmeta} WHERE meta_id = %d", $meta_id), ARRAY_A);
  if ($meta) {
    $data = maybe_unserialize($meta['meta_value']);
    if (is_array($data)) {
      $data['status'] = 'resolved';
      $wpdb->update($wpdb->postmeta, array('meta_value' => maybe_serialize($data)), array('meta_id' => $meta_id));
    }
  }
  wp_redirect(admin_url('admin.php?page=vpmr_reports'));
  exit;
});

add_action('admin_post_vpmr_delete', function () {
  if (! current_user_can('manage_options')) wp_die('permission');
  $meta_id = isset($_GET['meta_id']) ? intval($_GET['meta_id']) : 0;
  if (! wp_verify_nonce($_GET['_wpnonce'] ?? '', 'vpmr_delete_' . $meta_id)) wp_die('nonce');
  global $wpdb;
  $wpdb->delete($wpdb->postmeta, array('meta_id' => $meta_id));
  wp_redirect(admin_url('admin.php?page=vpmr_reports'));
  exit;
});

/* --- Admin: Settings Page --- */
function vpmr_settings_page()
{
  if (! current_user_can('manage_options')) wp_die('permission');
  $opts = get_option(VPMR_OPTION_KEY, vpmr_get_defaults());
  if (isset($_POST['vpmr_save'])) {
    check_admin_referer('vpmr_settings_save');
    $opts['spam_limit_count'] = max(1, intval($_POST['spam_limit_count']));
    $opts['spam_limit_window'] = max(60, intval($_POST['spam_limit_window']));
    $opts['webhook_url'] = sanitize_text_field($_POST['webhook_url'] ?? '');
    $opts['enable_webhook'] = isset($_POST['enable_webhook']) ? 1 : 0;
    update_option(VPMR_OPTION_KEY, $opts);
    echo '<div class="updated"><p>Saved.</p></div>';
  }
?>
  <div class="wrap">
    <h1>VIP PRO Movie Report Settings</h1>
    <form method="post">
      <?php wp_nonce_field('vpmr_settings_save'); ?>
      <table class="form-table">
        <tr>
          <th>Spam limit (count per IP per post)</th>
          <td><input type="number" name="spam_limit_count" value="<?php echo esc_attr($opts['spam_limit_count']); ?>" min="1"></td>
        </tr>
        <tr>
          <th>Spam window (seconds)</th>
          <td><input type="number" name="spam_limit_window" value="<?php echo esc_attr($opts['spam_limit_window']); ?>" min="60"></td>
        </tr>
        <tr>
          <th>Enable webhook alert</th>
          <td><label><input type="checkbox" name="enable_webhook" <?php checked($opts['enable_webhook'], 1); ?>> Enable</label></td>
        </tr>
        <tr>
          <th>Webhook URL</th>
          <td><input type="url" name="webhook_url" value="<?php echo esc_attr($opts['webhook_url']); ?>" style="width:100%"></td>
        </tr>
      </table>
      <p><button class="button button-primary" type="submit" name="vpmr_save">Save settings</button></p>
    </form>
  </div>
<?php
}

/* --- Optional: Shortcode to print button if you prefer shortcode usage --- */
add_shortcode('vpmr_button', function ($atts) {
  $atts = shortcode_atts(array('text' => 'Báo Lỗi'), $atts);
  return '<button class="action-btn report vpmr-open-report" id="reportBtn" aria-controls="vpmr-modal"><svg xmlns=\"http://www.w3.org/2000/svg\" width=\"24\" height=\"24\" viewBox=\"0 0 24 24\" fill=\"none\" stroke=\"currentColor\" stroke-width=\"2\" stroke-linecap=\"round\" stroke-linejoin=\"round\" class=\"feather feather-alert-circle\"><circle cx=\"12\" cy=\"12\" r=\"10\"></circle><line x1=\"12\" y1=\"8\" x2=\"12\" y2=\"12\"></line><line x1=\"12\" y1=\"16\" x2=\"12.01\" y2=\"16\"></line></svg><span class=\"count\">' . esc_html($atts['text']) . '</span></button>';
});
